#!/usr/bin/perl
#
# Rotates jpeg images $AMOUNT degrees clockwise. Moves originals to
# <name>_orig.jpg. 
#
# Copyright Arto Ters <ajt@iki.fi> 1999-2003
#
# Version: 1.12
# Licence: GNU General Public License (version 2 or any later version)
#          http://www.gnu.org/copyleft/gpl.txt
#
# Changelog:
#
# 1.0  Initial version
# 1.1  Automatic rotation feature and lossless rotation added.
#      Changed the output to interlaced (progressive) JPEG files 
#      in the case of lossy rotation. 
# 1.11 Fixed handling of filenames which contain spaces (thanks 
#      to Edward for finding the bug and providing a patch)
# 1.12 File name extension check fixed to accept capital letters
#
# Latest version: http://www.iki.fi/ajt/software/thumbnailtools/
######################################################################
#
# Global defaults:
# 
$AMOUNT = "auto";       # the amount to rotate, either keyword "auto" 
                        # or a number (degrees clockwise). 
                        # Amount "auto" rotates automatically images which 
                        # are marked as rotated in the EXIF header
                        # Can be adjusted with the command line option -a.

$QUALITY = "lossless";  # output image quality, either keyword "lossless"
                        # for lossless rotation (requires jpegtrans) or 
                        # a number, for example 75 for "JPEG 75% quality".
                        # Can be adjusted with the command line option -q

#
######################################################################

use Getopt::Std;

sub find_program {
    # Quick check if the given program can be found in the PATH
    # This will work only on Unix machines...
    my $program = shift;
    @system_dirs = split(':', $ENV{PATH});

    foreach (@system_dirs) {
	if (-x "$_/${program}") {
	    return 1;
	}
    }    
    # not found
    return 0;
}

# The main program
#
my %Options;
getopts('a:q:', \%Options);

if (@ARGV) 
{   
    if (exists($Options{"a"})) {
	$AMOUNT = $Options{"a"};
    }
    if (exists($Options{"q"})) {
	$QUALITY = $Options{"q"};
    }

    # Check that the necessary program(s) can be executed:
    if ($AMOUNT eq "auto") { # This needs jhead
	if (find_program("jhead") == 0) {
	    print STDERR "Couldn't find the \"jhead\" utility.\n\n";
	    print STDERR "Please make sure that it can be found in the PATH and try again.\n"; 
	    exit(1);
	}
    }

    # If we want lossless rotation, jpegtran (or jpegtran-mmx) does the job, 
    # otherwise use convert (old behaviour)
    my $jpegtran_name = "jpegtran-mmx"; # use the optimized one by default
    if ($QUALITY eq "lossless") {
	if (find_program("jpegtran-mmx") == 0) {
	    if (find_program("jpegtran") == 0) {
		print STDERR "Couldn't find the \"jpegtran\" utility.\n\n";
		print STDERR "Please make sure that it can be found in the PATH and try again.\n"; 
		exit(1);
	    }
	    else {
		$jpegtran_name = "jpegtran";
	    }
	}
    }
    else {
	if (find_program("convert") == 0) {
	    print STDERR "Couldn't find the \"convert\" utility (part of the ImageMagick package).\n\n";
	    print STDERR "Please make sure that it can be found in the PATH and try again.\n"; 
	    exit(1);
	}
    }
    
    my $image;
    foreach $image (@ARGV) {
	# Accept filenames ending in .jpg, .JPG, .jpeg and .JPEG
	if ($image =~ /(.*)\.([jJ][pP][eE]{0,1}[gG])/) {
	    print "Processing image ${image}";
	    my $basename = $1;
	    my $extension = $2;
	    my $origname = "$basename" . "_orig.${extension}";
	    my $rotate = 1;
	    my $degrees = $AMOUNT;

	    if (-f $origname) {
		print " ($origname already exists, using that!)";
	    }

	    if ($AMOUNT eq "auto") {
		# We'll use the original if it exists 
		# (made sense with convert but maybe not any more...)
		if (-f $origname) {
		    $jheadoutput = `jhead "$origname"`;
		}
		else {
		    $jheadoutput = `jhead "$image"`;
		}
		# By default no rotation if we don't find the field
		$rotate = 0;
		my @lines = split("\n", $jheadoutput);
		my $field;
		my $value;
		foreach $line (@lines) {
		    ($field, $value) = split(':', $line);
		    if ($field =~ /^Orientation/ && $value =~ /^ rotate /) {
			$degrees = scalar(substr($value, 8));
			if ($degrees > 0) {
			    $rotate = 1;
			}
		    }
		}
		if ($rotate == 1) {

		}
	    }
	    	   
	    if ($rotate == 1) {
		print ": rotating $degrees degrees.\n";

                # Let's preserve the original, if it doesn't exist
		if (! (-f $origname)) {
		    `mv "$image" "$origname"`;
		}
	    	    
		# Generate rotated image (lossless rotation by jhead or lossy with convert)
		if ($QUALITY eq "lossless") {
		    # Parameter -copy all necessary for not to lose the EXIF header info
		    `$jpegtran_name -rotate $degrees -copy all "$origname" >"$image"`;
		}
		else {
		    `convert -rotate $degrees -quality $QUALITY -interlace Line $origname $image`;
		}
	    }
	    else {
		print ": no rotation.\n";
	    }		    
	}
	else {
	    print "Skipping image $image\n";
	}
    }
}
else {
    print "Usage: rotate_images.pl [OPTIONS] IMAGEFILES (in jpeg format)\n\n";
    print "Available options:\n";
    print "-a degrees     the amount to rotate (clockwise, default $AMOUNT)\n"; 
    print "-q quality     the output jpeg quality (default ${QUALITY}%)\n"; 
    exit(1);
}
