#!/usr/bin/perl
#
# Compresses jpeg images to a preset quality. Moves originals to 
# <name>_orig.jpg. 
#
# Copyright Arto Ters <ajt@iki.fi> 1999-2003
#
# Version: 1.02
# Licence: GNU General Public License (version 2 or any later version)
#          http://www.gnu.org/copyleft/gpl.txt
#
# Changelog:
#
# 1.0  Initial version
# 1.01 Changed the output to interlaced (progressive) JPEG files.
# 1.02 File name extension check fixed to accept capital letters.
#
# Latest version: http://www.iki.fi/ajt/software/thumbnailtools/
######################################################################
#
# Global defaults:
# 
$QUALITY = 75;  # output image quality, can be adjusted with the 
                # command line option -q

$NO_COMPRESS_IF_ORIGINAL = 1;  # doesn't do the compression if there 
                               # already is a picture <name>_orig.jpg
                               # (not adjustable from command line)
#
######################################################################

use Getopt::Std;

sub find_program {
    # Quick check if the given program can be found in the PATH
    # This will work only on Unix machines...
    my $program = shift;
    @system_dirs = split(':', $ENV{PATH});

    foreach (@system_dirs) {
	if (-x "$_/${program}") {
	    return 1;
	}
    }    
    # not found
    return 0;
}

# The main program
#
getopts('q:');

if (@ARGV) 
{   
    if ($opt_q != 0) {
	$QUALITY = $opt_q;
    }

    # Check that the program convert can be executed:
    if (find_program("convert") == 0) {
	    print STDERR "Couldn't find the \"convert\" utility (part of the ImageMagick package).\n\n";
	    print STDERR "Please make sure that it can be found in the PATH and try again.\n"; 
	    exit(1);
    }

    my $image;
    my $skip;
    foreach $image (@ARGV) 
    {
	$skip = 0;
	# Accept filenames ending in .jpg, .JPG, .jpeg and .JPEG
	if ($image =~ /(.*)\.([jJ][pP][eE]{0,1}[gG])/) 
	{
	    my $basename = $1;
	    my $extension = $2;
	    my $origname = "$basename" . "_orig.${extension}";
	    
	    # Let's preserve the original, if it doesn't exist
	    if (-f $origname)
	    {
		if ($NO_COMPRESS_IF_ORIGINAL) 
		{
		    print "Skipping image $image\n";
		    $skip = 1;
		}
		else 
		{
		    print "Processing image $image\n";	
		    print "$origname already exists, using that!\n";
		}
	    }
	    else
	    {
		print "Processing image $image\n"; 
		`mv $image $origname`;
	    }
	    
	    # Generate compressed image
	    if (! $skip) 
	    {
		`convert -quality $QUALITY -interlace Line $origname $image`;
	    }
	}
	else
	{
	    print "Skipping image $image\n";
	}
    }
}
else
{
    print "Usage: compress_images.pl [OPTIONS] IMAGEFILES (in jpeg format)\n\n";
    print "Available options:\n";
    print "-q quality     the output jpeg quality (default ${QUALITY}%)\n"; 
    exit(1);
}
